#pragma once
#include "ATsystem.h"
#include "AT_filenames.h" //Though not needed by this file, it will likely be needed by the one using this.
//See Path_definitions_No.h for explanations
#include mergestring(path_definitions_,SYSTEM.h)
#include mergestring(ATfiles_,SYSTEM.h)
#ifndef SHRT_PATH //The ATfiles_SYSTEM.h has not defined SHRT_PATH
#include "shrt_path.h"
#endif

#define ATFILETYPE_ERROR AT_NODATA //Error en la determinación del tipo de fichero
#define ATFILETYPE_NOFILE 0
#define ATFILETYPE_FILE 1
#define ATFILETYPE_DIRECTORY 2
#define ATFILETYPE_FILE_PLAIN ATFILETYPE_FILE
#define ATFILETYPE_DIRECTORY_PLAIN ATFILETYPE_DIRECTORY
#define ATFILETYPE_LINK 4 //soft link
#define ATFILETYPE_FILE_LINK (ATFILETYPE_FILE | ATFILETYPE_LINK)
#define ATFILETYPE_DIRECTORY_LINK (ATFILETYPE_DIRECTORY | ATFILETYPE_LINK)

#define ATFILESYS_ERROR 1
#define ATFILESYS_BADPATH 2
#define ATFILESYS_ISNOT_FILE 3	//When it should be a file
#define ATFILESYS_ISNOT_DIR 4 //When it should be a directory

//The definition of these is opaque to the user (typically defined by each ATfiles_<SYSTEM>.h file)
struct strATFindFiles8;
struct strATFindFiles16;
typedef struct strATFindFiles8 ATFindFiles8;
typedef struct strATFindFiles16 ATFindFiles16;

#ifdef __cplusplus
extern "C"{
#endif
/*Transforma ruta en una ruta respecto al directorio en el que el SO busque los
módulos del ejecutable. path tiene que tener sitio suficiente para el resultado*/
void makepath_to_exec8(char8_t* path, const char8_t* ruta);
void makepath_to_exec16(char16_t* path, const char16_t* ruta);

/*These functions return:
		ATFILETYPE_ERROR: If the file does not exist or an error happens
		ATFILETYPE_NOFILE: La cadena de texto 'file' no es un archivo o directorio
		ATFILETYPE_FILE: If it is a file
		ATFILETYPE_DIRECTORY: If it is a directory

	For symbolic links the functions look at the file pointed to by the link,
	not at the link itself.
*/
u8int fileclass8(const char8_t* file);
u8int fileclass16(const char16_t* file);
u8int fileclass_utf8(const char8_t* file);
#if FILE8BITS_ARE_UTF8
#define fileclass_utf8(file) fileclass8(file)
#endif

/*Crear un directorio. Si ya existe, no hacer nada
	Return:
		0: El directorio se creó o ya existía
		ATFILESYS_BADPATH: Ruta incorrecta
		ATFILESYS_ISFILE: Ya hay un fichero con ese mismo nombre
		ATFILESYS_ERROR: No se puede crear
		AT_NOMEM: Solamente si tienen que convertir path a otra codificación y no hay memoria para ello.

	Para implementadores: La función puede devolver ATFILESYS_ERROR en cualquier
	caso de error, si no puede determinar con más precisión la causa.
*/
int ATCreateDirectory8(const char8_t* path);
int ATCreateDirectory16(const char16_t* path);
int ATCreateDirectory_utf8(const char8_t* path);
#if FILE8BITS_ARE_UTF8
#define ATCreateDirectory_utf8(path) ATCreateDirectory8(path)
#endif

/*Eliminar un fichero.
	Return:
		0: El fichero se eliminó correctamente o ya no existía
		ATFILESYS_BADPATH: Ruta incorrecta
		ATFILESYS_ISNOT_FILE: No es un fichero
		ATFILESYS_ERROR: No se puede eliminar
		AT_NOMEM: Solamente si tienen que convertir path a otra codificación y no hay memoria para ello.

	Para implementadores: La función puede devolver ATFILESYS_ERROR en cualquier
	caso de error, si no puede determinar con más precisión la causa.
*/
int ATDeleteFile8(const char8_t* path);
int ATDeleteFile16(const char16_t* path);
int ATDeleteFile_utf8(const char8_t* path);
#if FILE8BITS_ARE_UTF8
#define ATDeleteFile_utf8(path) ATDeleteFile8(path)
#endif

/*Para emplear estas funciones se ha de proceder de la siguiente manera:
    1º: Se llama a findnextfile_start<n> con el patrón de búsqueda.
		La función devuelve un puntero al objeto ATFindFiles<n> para la búsqueda.
    2º Se va llamando a findnextfile<n>. Esta función escribe en buffer el nombre del
		siguiente fichero cuyo nombre se ajusta a "pattern". Cuando buffer[0]
		sea '\0' hay que dejar de llamar (v. infra).
    3º Cuando hemos terminado hay que llamar a findnextfile_end<n>

    La función findnextfile_start<n> devuelve NULL si no hay memoria. En cualquier
otro caso devuelve un puntero que habrá que liberar llamando a findnextfile_end<n>.
Si se produce algún error en findnextfile_start<n> (por ejemplo, que pattern no sea
una cadena de texto válida) una llamada a ATFindFiles<n>_isOK pasándole el
puntero devuelto por findnextfile_start<n> devolverá falso. No pasa nada por emplear
esta estructura en findnextfile<n>; simplemente no encontrará ningún fichero y
devolverá !=0.
    Las funciones findnextfile<n> y findnextfile_end<n> devuelven !=0 si hay algún
error. En la función findnextfile<n>, en caso de error se pondrá buffer[0]='\0';
Si no hay más ficheros también se devolverá buffer[0]='\0', pero el valor devuelto por
la función sera 0.
*/
bint ATFindFiles8_isOK(const ATFindFiles8 *find_files);
ATFindFiles8* findnextfile_start8(const char8_t* pattern);
int findnextfile8(ATFindFiles8 *find_files, char8_t buffer[SHRT_PATH]);
int findnextfile_end8(ATFindFiles8 *find_files);

bint ATFindFiles16_isOK(const ATFindFiles16 *find_files);
ATFindFiles16* findnextfile_start16(const char16_t* pattern);
int findnextfile16(ATFindFiles16 *find_files, char16_t buffer[SHRT_PATH]);
int findnextfile_end16(ATFindFiles16 *find_files);

/*Obtiene todos los ficheros cuyo nombre se ajusta a "pattern". Un NULL indica
el final del array de punteros. Si el primer elemento ya es NULL es que ningún
fichero se ajusta al patrón indicado. Si el puntero en sí devuelto es NULL es que
se produjo algún error.
    El contenido de cada fichero; es decir, el array de char8_t* apuntado por cada
puntero del array devuelto, es modificable, pero no se puede escribir más allá del
'\0' que marca el final del mismo.
    El puntero devuelto (en caso de que sea !=NULL), hay que liberarlo mediante la
función free_filelist<n>*/
char8_t **findfiles8(const char8_t* pattern);
char16_t **findfiles16(const char16_t* pattern);
void free_filelist8(char8_t* *filelist);
void free_filelist16(char16_t* *filelist);

#ifdef __cplusplus
}
#endif

/*SHRT_PATH / SHRT_DIR limitted functions, unless overriden by the <SYSTEM> implementation:

findfiles8/16: Cada nombre de fichero no debe ocupar más de SHRT_PATH. Nótese no obstante
	que esto no incluye la ruta, solamente el nombre del fichero.
*/

//Because the default is to have them defined, the file
//#include "ATfiles_undef.h"
//is provided for undefining them.
#if FILE_PREFERRED_ENCODING==FILE_ENCODING_BIT8
#include "ATfiles_define8.h"
#else
#include "ATfiles_define16.h"
#endif
