//Some utilities parse words on a string without any need for the structures defined here.
//They are defined in ATparsestring.h
#include "ATparsestring.h"

//This file is not intended to be included standalone. It needs the definitions of ATfileinput.h.
//Hence, it is not protected against double inclussion (which would be harmless anyway)

/*Las macros con alguna mayúscula en su nombre, por ejemplo if_Moreinl frente a if_moreinl tienen
en cuenta que existe un caracter, COMMENT_CHAR, que introduce un comentario que dura hasta
el final de línea. Se pueden usar estas macros aunque no haya comentarios sin más que definir
COMMENT_CHAR como '\n'. If this is so these macros behave as if there were no comment char.
Las macros sin mayúscula podrían por tanto omitirse por completo. Se pueden emplear si sabemos
que no existe carácter de comentario para optimizar la lectura del fichero o para que se manifestar
que no existe un COMMENT_CHAR. */

#ifndef COMMENT_CHAR
#define COMMENT_CHAR '\n'
#endif

/* The following macros are the only ones that need to be redefined to keep track of the line
number. Here we make them default to 'plain', which counts not. */
#undef advance
#undef finishline
#include "fileinput_plain.h"

/******---------        Macros that do not get the name of a function to be called        ---------******/

//The macro advance(buffer) moves pc to the beginning of the next word. This is typically used as
//the first instruction when reading a file.

//Avanzar hasta la palabra siguiente y/0 el final de línea, o quedarse (stay)
#define advance_PLAIN(buffer) str_no_stn((buffer).pc)
#define advanceinline(buffer) str_no_st((buffer).pc)

#define gotoendofline(buffer) if(*(buffer).pc!='\0'){while(*(buffer).pc!='\n') (buffer).pc++;}
#define finishline_PLAIN(buffer) if(*(buffer).pc!='\0'){while(*(buffer).pc!='\n') (buffer).pc++; (buffer).pc++;}	/*while(*pc++); may be better for other processors*/
#define finishline_stay(buffer) finishline(buffer)
#define finishline_advance(buffer) finishline(buffer); advance(buffer)
#define finishline_advanceinline(buffer) finishline(buffer); advanceinline(buffer)

#define if_moreinl(buffer) if(*(buffer).pc!='\n')
#define if_nomore(buffer) if(*(buffer).pc=='\n')
#define iflike_moreinl(buffer) iflike(*(buffer).pc!='\n')
#define ifunlike_nomore(buffer) ifunlike(*(buffer).pc=='\n')

//Lo mismo, para cuando existe un COMMENT_CHAR
#define Advance(buffer) advance(buffer); while(*(buffer).pc==COMMENT_CHAR){finishline_advance(buffer);}
#define Advanceinline advanceinline
#define finishline_Advance(buffer) finishline(buffer); Advance(buffer)

#define MORE_INL_pc(pc) (*(pc)!='\n' && *(pc)!=COMMENT_CHAR)
#define NOMORE_INL_pc(pc) (*(pc)=='\n' || *(pc)==COMMENT_CHAR)
#define MORE_INL(buffer) MORE_INL_pc((buffer).pc)
#define NOMORE_INL(buffer) NOMORE_INL_pc((buffer).pc)
#define if_Moreinl(buffer) if(MORE_INL(buffer))
#define if_Nomore(buffer) if(NOMORE_INL(buffer))
#define iflike_Moreinl(buffer) iflike(MORE_INL(buffer))
#define ifunlike_Nomore(buffer) ifunlike(NOMORE_INL(buffer))

#define Gotoendofline(buffer) if(*(buffer).pc!='\0'){while(MORE_INL(buffer)) (buffer).pc++;}

//Pass over a word y avanzar tras ella. Sin comment-char
#define ignore(buffer) if(*(buffer).pc!='\0'){str_stn((buffer).pc);}
#define ignore_stay(buffer) ignore(buffer)
#define ignore_advance(buffer) if(*(buffer).pc!='\0'){str_stn((buffer).pc); advance(buffer);}
#define ignore_advanceinline(buffer) if(*(buffer).pc!='\0'){str_stn((buffer).pc); advanceinline(buffer);}

//Si el carácter de comentario en medio de la palabra se considera parte de la palabra
#define ignore_Stay(buffer) ignore(buffer)
#define ignore_Advance(buffer) ignore(buffer); Advance(buffer)
#define ignore_Advanceinline(buffer) ignore(buffer); Advanceinline(buffer)

//El caracter de comentario termina la palabra
#define Ignore(buffer) if(*(buffer).pc!='\0'){str_stnC((buffer).pc);}
#define Ignore_Stay(buffer) Ignore(buffer)
#define Ignore_Advance(buffer) if(*(buffer).pc!='\0'){str_stnC((buffer).pc); Advance(buffer);}
#define Ignore_Advanceinline(buffer) if(*(buffer).pc!='\0'){str_stnC((buffer).pc); Advanceinline(buffer);}

/* Macros for preparing a word or a line for further processing */

#define savecharnext(buffer) (buffer).savedchar=*(buffer).next, *(buffer).next='\0'
#define savecurchar(buffer) (buffer).next=(buffer).pc, (buffer).savedchar=*(buffer).next, *(buffer).next='\0'

#define prepareline(buffer) if(*((buffer).next=(buffer).pc)!='\0'){while(*(buffer).next!='\n') (buffer).next++;} savecharnext(buffer)
#define Prepareline(buffer) if(*((buffer).next=(buffer).pc)!='\0'){while(MORE_INL_pc((buffer).next)) (buffer).next++;} savecharnext(buffer)

//Prepara una línea, poniendo un '\0' en el primer ' ' o '\t' de una serie de tales caracteres
//al final de la línea. Si no hay ninguno el '\0' se pone sobre el '\n' o COM._C. del final de línea
#define Prepare_line(buffer) \
	if(*((buffer).next=(buffer).pc)!='\0'){while(MORE_INL_pc((buffer).next)) (buffer).next++;\
	while(*--(buffer).next==' ' || *(buffer).next=='\t'); (buffer).next++;}\
	savecharnext(buffer)

//Prepara una palabra, poniendo un '\0' en el stn o stnC que la cierra.
//Do not add a buffer.next++ inside the if, before str_stn.
#define advance_next(buffer) (buffer).next=(buffer).pc; if(*(buffer).pc!='\0'){str_stn((buffer).next);}
#define next_fwd_save(buffer) advance_next(buffer); savecharnext(buffer)
#define Advance_next(buffer) (buffer).next=(buffer).pc; if(*(buffer).pc!='\0'){str_stnC((buffer).next);}
#define Next_fwd_save(buffer) Advance_next(buffer); savecharnext(buffer)

/*These, together with the resume series, are typically for error messages or error recovery.
next>pc can only happen after a call to if_... that failed; next==pc can only happen after
a call to getslow of if_getslow. In both cases buffer.next points to the end of the word and
*buffer.next=savedchar would have just been executed if needed*/
#define go_back_pc(pc) while(*--(pc)>' ' || (*(pc)!=' ' && *(pc)!='\t' && *(pc)!='\n' && *(pc)!='\0')); (pc)++
#define go_back(buffer) go_back_pc((buffer).pc)
#define isolate_word_delimited(buffer,limit_macro) \
	if(*(buffer).pc=='\0') (buffer).next=(buffer).pc;\
	else if((buffer).next<(buffer).pc){(buffer).next=(buffer).pc; limit_macro((buffer).next);}\
	savecharnext(buffer); go_back(buffer)
#define isolate_word(buffer) isolate_word_delimited(buffer,str_stn)
#define Isolate_word(buffer) isolate_word_delimited(buffer,str_stnC)

#define restorecharnext(buffer) *(buffer).next=(buffer).savedchar
#define resume(buffer) *(buffer).next=(buffer).savedchar, (buffer).pc=(buffer).next
#define resume_stay(buffer) resume(buffer)
#define resume_advance(buffer) resume(buffer); advance(buffer)
#define resume_advanceinline(buffer) resume(buffer); advanceinline(buffer)

#define resume_Stay(buffer) resume(buffer)
#define resume_Advance(buffer) resume(buffer); Advance(buffer)
#define resume_Advanceinline(buffer) resume(buffer); Advanceinline(buffer)
/*End of that*/


/******---------        Macros that call a function to process an input word        ---------******/

/*	The concept behind these macros is that they get the next word and use 'func' to process
it, the returned value is stored at 'x' and finally buffer.pc is advanced to the beginning of the
next word or the '\0' signalling the end of the file if there are no more words.
	A word is something delimited by one of ' ', '\t', or '\n' at each side (to be called blanks
hereafter). The function that loads the file takes care to put one '\n' at the end, just before
the '\0', if there is none.
	The ending of a line often needs special treatment, so there are four kinds of macros according
to how they advance pc towards the next word. These kinds are distinguished by the ending
of the macro name. There are four possible suffixes:
	-- _stay. pc is kept at the blank that ends the word.
	-- _advance. pc is advanced to the beginning of the next word or to '\0' if there is none.
	-- _advanceinline. Like the previous, but a '\n' also stops pc.
	-- _advancecare. Like _advance, but if a '\n' is passed while advancing it is saved in savedchar.
		Otherwise the character at savedchar will be a different one.

	After a call of one _advancecare macro you will use the macro ifinline or ifnlpassed, otherwise
you could have spared the 'care'.
	The main macros are advance and the nocheck_get, if_get and ifnot_get series. They are
invoked like this:

	nocheck_get_<suffix>(buffer, x, func)
	if_get_<suffix>(buffer, x, func)			ifnot_get_<suffix>(buffer, x, func)

where <suffix> is one of the four possible actions and the 'func' a function the form of which is
explained below. For example:

	Bufferti8 buffer;
	uint nret;
	double time;

	nret=tiopen8(&buffer,"datos.txt");
	ifunlike(nret>ATFILEI_MAXSIZE) return -(int)(-nret);
	nret=0;

	advance(buffer);
	ifnot_get_advance(buffer, time, vfdouble___str8){nret=1; goto error;}

	The final instruction when these macros are expanded is an if, which for the if_get series
is if_tibuf_ok(buffer) and for the ifnot_get series is ifnot_tibuf_ok(buffer), and they are opposites
of each other.
	There is no final if for the nocheck series. Since usually errors are not to be ignored, these
are intended to be used in switch cases, where the test is performed after the switch block,
or if a test including something else in addition to NOT_TIBUF_OK (typically) is wanted
after the reading. Either if_tibuf_ok or ifnot_tibuf_ok or an if satatement including
TIBUF_OK or NOT_TIBUF_OK should be called afterwards.

	In the example above, if the reading was successfull pc is left pointing to the beginning of
the next word or to the '\0' at the end. If one of the suffixes _stay _advanceinline or _advancecare
had been used in place of _advance pc would be placed accdordingly. You need not check for the
end of the file after each call. The macros will never move .pc (nor .next) beyond the end of
the file. A common technic consists in reading all but one of the words that you expect to be
present and then check for the end of the file, viz. if(*buffer.pc=='\0').
	Nor do you need to check for the end of the line after every call with an _advanceinline suffix.
The macros if_moreinl and if_nomre are provided fot testing the end of the line. They expand just
to if(*(buffer).pc!='\n') and if(*(buffer).pc=='\n') respectively.

	The function 'func' must have this prototype: <type> func(const char8_t **), or with char16_t
instead of char8_t for the Bufferti16, since it will be called like this:

									x=func(&buffer.pc);

You may not respect the const, but your functions must not convert non-blanks into blanks or
vice-versa, let alone create or destroy '\0'.
	buffer.pc shall be advanced to point to the position of the character that stopped the reading.
Then the macro will check whether it is a blank or '\0'. If so the if(...) that end these macros will
test to true for the if_get series and to false for the ifnot_get ones; otherwise, buffer.next will be
positioned at the blank that ends the word and the if(...) will test to the opposite of the previous.
This is why the ifnot_get series is used more often than the if_get ones, for ifnot_get tests to true
normally some action is performed.
	In case of a reading error you may call go_back(buffer) to move pc back to the beginning of
the word, i.e. to the first character of the word, and read again with a different function, or you
may use isolate_word, which in addition to moving .pc back moves .next to the blank that
ends the word, then saves the blank pointed by .next and writes a '\0' there. It is intended to allow
the use of .pc as a string to report an error. In order to continue after a call to isolate_word you must
call resume in one of its variants.

	A small second set of macros is provided in case there are comments introduced by a comment
char and extending till the end of the line:

	Advance, Advanceinline, finishline_Advance, if_Nomore, if_Nomore &c.

all with a capital letter within it.

    Strings are handled differently. See below
*/

/*Most of the series that follow, of the form <macroseriesname>_<suffix> could have been defined
with 1/4 of the code as exemplified here with finishline:
	#define finishline_do(action,buffer) finishline(buffer); action(buffer);

but the current set of definitions is more natural when used. But in case the user wants to take
that approach for macros of his the macro stay is defined, as well as the _stay macros which are
equal to the same ones without the _stay.
*/

#define cast_adpc(buffer) cast_pc_addr((buffer).pc)

//stay is too common a word to have it defined here uncondinionally
#ifdef stay
Diagnostic_Message("The macro stay is already defined. Use atfi_stay instead")
#else
#define stay(buffer)
#endif
#define atfi_stay(buffer)

/****----        The function 'func' admits words ended by any of ' ','\t', '\n' and, maybe, COMMENT_CHAR        ----****/

/* Macros that parse the word(s) and proceed if the pointer advanced to the end-of-word upon parsing */
#define getfast(buffer,x,func) x=func(cast_adpc(buffer))
#define getfast_advance(buffer,x,func) getfast(buffer,x,func); advance(buffer)
#define getfast_advanceinline(buffer,x,func) getfast(buffer,x,func); advanceinline(buffer)
#define getfast_Advance(buffer,x,func) getfast(buffer,x,func); Advance(buffer)
#define getfast_Advanceinline(buffer,x,func) getfast(buffer,x,func); Advanceinline(buffer)

/*In the following series, if the function fails buffer.next will be advanced to the end of the word.*/
#define TIBUF_OK(buffer) (buffer).pc>=(buffer).next
#define NOT_TIBUF_OK(buffer) (buffer).pc<(buffer).next
#define if_tibuf_ok(buffer) iflike(TIBUF_OK(buffer))
#define ifnot_tibuf_ok(buffer) ifunlike(NOT_TIBUF_OK(buffer))

/*Internal. Do not use them. These are pieces for building other macros*/

//Place next at the end of word
#define get_value_ptr(buffer,x,func) \
	next_fwd_save(buffer); x=func(cast_adpc(buffer))

#define Get_value_ptr(buffer,x,func) \
	Next_fwd_save(buffer); x=func(cast_adpc(buffer))

#define _nocheck_get(action,buffer,x,func) \
	x=func(cast_adpc(buffer));\
	if(is_stn(*(buffer).pc)){action(buffer);}\
	else{advance_next(buffer);}

#define _internal_ifnot_get(action,buffer,x,func) _nocheck_get(action,buffer,x,func) ifnot_tibuf_ok(buffer)
#define _internal_if_get(action,buffer,x,func) _nocheck_get(action,buffer,x,func) if_tibuf_ok(buffer)

#define _nocheck_Get(action,buffer,x,func) \
	x=func(cast_adpc(buffer));\
	if(is_stnC(*(buffer).pc)){action(buffer);}\
	else{Advance_next(buffer);}
#define _internal_ifnot_Get(action,buffer,x,func) _nocheck_Get(action,buffer,x,func) ifnot_tibuf_ok(buffer)
#define _internal_if_Get(action,buffer,x,func) _nocheck_Get(action,buffer,x,func) if_tibuf_ok(buffer)
/* End of internal */

/* Macros that do not end in if_tibuf_ok() or ifnot_tibuf_ok(). Notwithstanding their names
they do check for failure and if so advance next to the end-of-word. They just are missing the
final if(not)_tibuf_ok in contrats to the series that follow.*/
#define nocheck_get_stay(buffer, x, func) _nocheck_get(atfi_stay,buffer,x,func)
#define nocheck_get_advance(buffer, x, func) _nocheck_get(advance,buffer,x,func)
#define nocheck_get_advanceinline(buffer, x, func) _nocheck_get(advanceinline,buffer,x,func)
#define nocheck_get_Stay(buffer, x, func) _nocheck_Get(atfi_stay,buffer,x,func)
#define nocheck_get_Advance(buffer, x, func) _nocheck_Get(Advance,buffer,x,func)
#define nocheck_get_Advanceinline(buffer, x, func) _nocheck_Get(Advanceinline,buffer,x,func)

/* Macros that check for failure after the function call */
#define ifnot_get_stay(buffer, x, func) _internal_ifnot_get(atfi_stay,buffer,x,func)
#define ifnot_get_advance(buffer, x, func) _internal_ifnot_get(advance,buffer,x,func)
#define ifnot_get_advanceinline(buffer, x, func) _internal_ifnot_get(advanceinline,buffer,x,func)
#define ifnot_get_Stay(buffer, x, func) _internal_ifnot_Get(atfi_stay,buffer,x,func)
#define ifnot_get_Advance(buffer, x, func) _internal_ifnot_Get(Advance,buffer,x,func)
#define ifnot_get_Advanceinline(buffer, x, func) _internal_ifnot_Get(Advanceinline,buffer,x,func)

/* Macros that check for succees after the function call */
#define if_get_stay(buffer, x, func) _internal_if_get(atfi_stay,buffer,x,func)
#define if_get_advance(buffer, x, func) _internal_if_get(advance,buffer,x,func)
#define if_get_advanceinline(buffer, x, func) _internal_if_get(advanceinline,buffer,x,func)
#define if_get_Stay(buffer, x, func) _internal_if_Get(atfi_stay,buffer,x,func)
#define if_get_Advance(buffer, x, func) _internal_if_Get(Advance,buffer,x,func)
#define if_get_Advanceinline(buffer, x, func) _internal_if_Get(Advanceinline,buffer,x,func)
/*End of the standard series*/


/****----        The function 'func' needs the word to be terminated by '\0'        ----****/

/* 'func' takes a char_8(16)* */
#define getslow(buffer, x, func) \
	next_fwd_save(buffer);\
	x=func((buffer).pc);\
	*(buffer).next=(buffer).savedchar; (buffer).pc=(buffer).next

#define getslow_stay(buffer, x, func) getslow(buffer,x,func)
#define getslow_advance(buffer, x, func) \
	next_fwd_save(buffer);\
	x=func((buffer).pc); (buffer).pc=(buffer).next;\
	if((buffer).savedchar!='\0'){(buffer).pc++; advance(buffer);}
#define getslow_advanceinline(buffer, x, func) getslow_stay(buffer,x,func); advanceinline(buffer)

#define Getslow(buffer, x, func) \
	Next_fwd_save(buffer);\
	x=func((buffer).pc);\
	*(buffer).next=(buffer).savedchar; (buffer).pc=(buffer).next

#define Getslow_Stay(buffer, x, func) Getslow(buffer,x,func)
#define Getslow_Advance(buffer, x, func) \
	Next_fwd_save(buffer);\
	x=func((buffer).pc); (buffer).pc=(buffer).next;\
	if((buffer).savedchar!='\0'){(buffer).pc++; Advance(buffer);}
#define Getslow_Advanceinline(buffer, x, func) Getslow_Stay(buffer,x,func); Advanceinline(buffer)
/*End of that*/

/* 'func' takes a pointer to a char_8(16)* which advances */

/* internal */
#define _internal_ifnot_getslow(action,buffer, x, func) \
	get_value_ptr(buffer, x, func);\
	if((buffer).savedchar!='\0'){\
		*(buffer).next=(buffer).savedchar;\
		if((buffer).pc==(buffer).next){action(buffer);}\
	}\
	ifnot_tibuf_ok(buffer)
#define _internal_if_getslow(action,buffer, x, func) \
	get_value_ptr(buffer, x, func);\
	if((buffer).savedchar!='\0'){\
		*(buffer).next=(buffer).savedchar;\
		if((buffer).pc==(buffer).next){action(buffer);}\
	}\
	if_tibuf_ok(buffer)

/*user macros*/
#define ifnot_getslow_stay(buffer,x,func) _internal_ifnot_getslow(atfi_stay,buffer,x,func)
#define ifnot_getslow_advance(buffer,x,func) _internal_ifnot_getslow(advance,buffer,x,func)
#define ifnot_getslow_advanceinline(buffer,x,func) _internal_ifnot_getslow(advanceinline,buffer,x,func)
//
#define if_getslow_stay(buffer,x,func) _internal_if_getslow(atfi_stay,buffer,x,func)
#define if_getslow_advance(buffer,x,func) _internal_if_getslow(advance,buffer,x,func)
#define if_getslow_advanceinline(buffer,x,func) _internal_if_getslow(advanceinline,buffer,x,func)
/*End of that*/


/*	Strings

In order to use the string by passing it to a function that shall copy it to another location,
compare it against another string or analyse it in whatever way, e.g., strlen(), it is necessary
to have it ended by '\0'. For this purpose the special macro prepare_string is provided, which
seeks the end of the word, places a '\0' there, saving the previous character at savedchar, and
leaves pc pointing to the beginning of the word (therefore, the situation of buffer after a call
to prepare_string is identical than after a call to isolate word). Thus .pc is a C string that can be
passed to any function expecting a string. After using pc for whatever be needed, the next call
for continuing with the reading of the file must be resume in one of the available endings:
resume, resume_advance and resume_advanceinline. Note that resume_advancecare is not available.
	If you are going to use resume_advance or you'd like to use resume_advancecare
you may use the shortcuts
	string_advance(buffer,s)      string_advancecare(buffer,s)
in place of both prepare_string and the subsequent resume. After these calls you may
use s as you would use pc had you called prepare_string, and you don't need to call resume
afterwards, but the extra variable s is needed.
	Of course, you can provide your own func that stops at a blank, copies the word to somewhere
else and returns a pointer to the copy, and use the nocheck/if/ifnot_get_<suffix> macro series
as with integers or floats.

	If the string is not going to be stored anywhere but is just being compared to other
NULL-ended string(s), as when looking for key names, the fuctions pccpm8 and pccmp16
above can be used. Thereby the use of prepare_string is avoided, which is delicate because any
reading afterwards must be preceeded by resume. After pccpm8 you likely call one of the
ignore_<suffix> series.
*/

#define prepare_string(buffer) next_fwd_save(buffer)
#define string_advance(buffer,s) getslow_advance(buffer,s,)
//No existe string_advanceinline porque s tiene que quedar apuntando a una
//palabra terminada por '\0', mientras que si el caracter que había allí era '\n', pc
//tendría que quedar ahí y tendría que ser *pc='\n', y ambas cosas a la vez son imposibles.

#define Prepare_string(buffer) Next_fwd_save(buffer)
#define String_Advance(buffer,s) Getslow_Advance(buffer,s,)
/*end strings*/


/** Macros that remeber if an e.o.l. was passe while advancing, by storing it in savedchar **/

#define advancecare(buffer) advanceinline(buffer); if(((buffer).savedchar=*(buffer).pc)=='\n') advance(buffer)
#define finishline_advancecare(buffer) finishline(buffer); advancecare(buffer)
#define ifinline(buffer) if((buffer).savedchar!='\n')
#define ifnlpassed(buffer) if((buffer).savedchar=='\n')
#define ifinline_finishline(buffer) ifinline(buffer){finishline(buffer);}
#define ignore_advancecare(buffer) if(*(buffer).pc!='\0'){str_stn((buffer).pc); advancecare(buffer);}
#define resume_advancecare(buffer) resume(buffer); advancecare(buffer)
#define getfast_advancecare(buffer,x,func) getfast(buffer,x,func); advancecare(buffer)
#define nocheck_get_advancecare(buffer, x, func) _nocheck_get(advancecare,buffer,x,func)
#define ifnot_get_advancecare(buffer, x, func) _internal_ifnot_get(advancecare,buffer,x,func)
#define if_get_advancecare(buffer, x, func) _internal_if_get(advancecare,buffer,x,func)
#define getslow_advancecare(buffer, x, func) getslow_stay(buffer,x,func); advancecare(buffer)
#define ifnot_getslow_advancecare(buffer,x,func) _internal_ifnot_getslow(advancecare,buffer,x,func)
#define if_getslow_advancecare(buffer,x,func) _internal_if_getslow(advancecare,buffer,x,func)
#define string_advancecare(buffer,s) getslow_advancecare(buffer,s,); *(buffer).next='\0'
