#pragma once
#include <ATcrt/ATiflike.h>
#include <ATcrt/AT_memutils.h>

/*inner_loopN denotes an inner_loopn where N is a compile-time constant.
The definitions are very likely to be the same, or the difference very little, so
there is no provision for an extra variant in the macros that use it. inner_loopN
is used when the macro is for some other reason written to receive a constant
value for n.*/
//0 means that _inner_loop_ may equal zero at the beginning; that is, n may be zero
#define _inner_loop_ merge(_inner_loop_,__LINE__)
#define inner_loopN(N,code) cint _inner_loop_=N; do{_inner_loop_--; code;} while(_inner_loop_)
#define inner_loopn(n,code) cint _inner_loop_=n; do{_inner_loop_--; code;} while(_inner_loop_)
#define inner_loop0(n,code) cint _inner_loop_=n; while(_inner_loop_){_inner_loop_--; code;}

#define zeroset_type(Type,x,size) merge(zeroset_,Type)(x,size)

/*Macros para multiplicar matrices.

    A=B*C. Los dos sufijos 1/n indican si las matrices B y C se multiplican por filas
o por columnas. Una multiplicación normal de matrices sería mmul_1n.
Dimensiones:  A: a*c, B: a*b (ó b*a), C: b*c (ó c*b)
Un desplazamiento en C en el sentido en el que no se multiplica implica avanzar
1 en la matriz del resultado, mientras que un desplazamiento análogo en B implica
avanzar c.
    El sufijo BC o AC indica que las macros están pensadas para ser llamads con valores
constantes para ambas dimensiones en el momento de compilación. Se supone
que siendo eso así son más eficientes que las que no tienen sufijo.
    Las macros con sufijo open deben dejar avanzados los punteros exepcto el que
se inicializa para cada iteración del bucle exterior
*/
/*The C preprocessor is the powerless macro language ever.
I only needed to check if OP begins by '=', or simply to check whether
it is either of "=", "=+", "=-", but that is impossible. It can be done by code:
{int ii=1; if(!(ii OP 0)<code>}, or if(#OP[0]=='='), but not by the preprocessor.
Therefore the duplication _equal and _plus is needed.
In order to provide an example, here is the definition of mmult_1n in case
#begindef/#endef existed

#begindef mmul_1n(TypeR,A,OP,B,C,a,b,c)
	#if #OP[0]=='='
	mmul_1n_equal(TypeR,A,OP,B,C,a,b,c)
	#else
	mmul_1n_plus(TypeR,A,OP,B,C,a,b,c)
	#endif
#enddef
*/
/*Finalmente, las macros pueden comprobar previamente que las variables de los
bucles internos (dos de entre a, b y c) no son cero. Si se omite la comprobación se
ahorran muchos condicionales, pero el usuario tiene que estar seguro de que se
llaman con valores distintos de cero, de lo contrario la "función" se quedará colgada.
Las variantes _open son para una optimización mayor y no tienen esa posibilidad.
Se pueden omitir las comprobaciones si de define NO_ZERO_CHECK previamente
a la inclusión de este fichero.
*/

#ifdef NO_ZERO_CHECK
	#define ZEROCHECK_ac(a,c)
	#define ZEROCHECK_a(a)
#else
	#ifndef iflike
	#define iflike if
	#endif
	#define ZEROCHECK_ac(a,c) iflike((a)!=0 && (c)!=0)
	#define ZEROCHECK_a(a) iflike((a)!=0)
#endif
#define ZEROCHECK_bc ZEROCHECK_ac
#define ZEROCHECK_b ZEROCHECK_a

/*Escribo primero las macros para multiplicar por un vector. Solamente hay dos casos
posibles: que la otra matriz se multiplique por filas o por columnas*/
#define mmul_vector1_open(TypeR,pA,OP,pB,pC,a,c) \
	cint it=a;\
	do{ it--;\
		TypeR aux=0;\
		inner_loopn(c, aux+=*pB++**pC++);\
		*pA++ OP aux;\
		pC-=c;\
	}while(it);
#define mmul_vector1(TypeR,A,OP,B,v,a,c) \
	ZEROCHECK_ac(a,c){TypeR *pA=A, *pB=B, *pC=v;\
		mmul_vector1_open(TypeR,pA,OP,pB,pC,a,c) \
	}

#define mmul_vectorn_equal(TypeR,A,OP,B,v,a,c) \
	zeroset_type(TypeR,A,a)\
	ZEROCHECK_ac(a,c){TypeR *pB=B, *pC=v, *pA=A;\
	cint it=c;\
	do{ it--;\
		TypeR aux=*pC++;\
		inner_loopn(a, *pA++ +##OP *pB++*aux);\
		pA-=a;\
	}while(it);\
	}
#define mmul_vectorn_plus_open(TypeR,pA,OP,pB,pC,a,c) \
	cint it=c;\
	do{ it--;\
		TypeR aux=*pC++;\
		inner_loopn(a, *pA++ OP *pB++*aux);\
		pA-=a;\
	}while(it);
#define mmul_vectorn_plus(TypeR,A,OP,B,v,a,c) \
	ZEROCHECK_ac(a,c){TypeR *pB=B, *pC=v, *pA=A;\
		mmul_vectorn_plus_open(TypeR,pA,OP,pB,pC,a,c) \
	}

#define mmul_11(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_bc(b,c){TypeR *pA=A;\
	TypeR *pB=B;\
	for(cint it=a;it;pB+=b){ it--;\
		TypeR *pC=C;\
		cint jt=c;\
		do{ jt--;\
			TypeR aux=0;\
			inner_loopn(b, aux+=*pB++**pC++);\
			*pA OP aux;\
			pB-=b, pA++;\
		}while(jt);\
	}}

#define mmul_11_BC_open(TypeR,pA,OP,pB,pC,a,b,c) \
	for(cint it=a;it;pB+=b){ it--;\
		cint jt=c;\
		do{ jt--;\
			TypeR aux=0;\
			inner_loopN(b, aux+=*pB++**pC++);\
			*pA OP aux;\
			pB-=b, pA++;\
		}while(jt);\
		pC-=b*c;\
	}
#define mmul_11_BC(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_bc(b,c){TypeR *pA=A, *pB=B, *pC=C;\
		mmul_11_BC_open(TypeR,pA,OP,pB,pC,a,b,c)\
	}

#define simmul_11_open(TypeR,pA,OP,TypeRb,pB,pC_o,a,b) \
	for(cint it=a;it;pB+=b,pC_o+=b){\
		TypeRb *pC=pC_o;\
		cint jt=it--;\
		do{ jt--;\
			TypeR aux=0;\
			inner_loopn(b,aux+=*pB++**pC++);\
			*pA OP aux;\
			pB-=b, pA++;\
		}while(jt);\
		pA+=a-it;\
	}
#define simmul_11(TypeR,A,OP,TypeRb,B,C,a,b) \
	ZEROCHECK_b(b){TypeR *pA=A;\
		TypeRb *pB=B, *pC_o=C;\
		simmul_11_open(TypeR,pA,OP,TypeRb,pB,pC_o,a,b)\
	}

#define mmul_1n_equal(TypeR,A,OP,B,C,a,b,c) \
	zeroset_type(TypeR,A,a*c)\
	ZEROCHECK_bc(b,c){TypeR *pA=A;\
	TypeR *pB=B;\
	for(cint it=a;it;pA+=c){ it--;\
		TypeR *pC=C;\
		cint jt=b;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopn(c, *pA++ +##OP aux**pC++);\
			pA-=c;\
		}while(jt);\
	}}
#define mmul_1n_equal_BC(TypeR,A,OP,B,C,a,b,c) \
	zeroset_type(TypeR,A,a*c)\
	ZEROCHECK_bc(b,c){TypeR *pA=A, *pB=B, *pC=C;\
	for(cint it=a;it;pA+=c){ it--;\
		cint jt=b;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopN(c, *pA++ +##OP aux**pC++);\
			pA-=c;\
		}while(jt);\
		pC-=b*c;\
	}}

#define mmul_1n_plus(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_bc(b,c){TypeR *pA=A;\
	TypeR *pB=B;\
	for(cint it=a;it;pA+=c){ it--;\
		TypeR *pC=C;\
		cint jt=b;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopn(c, *pA++ OP aux**pC++);\
			pA-=c;\
		}while(jt);\
	}}
#define mmul_1n_plus_BC_open(TypeR,pA,OP,pB,pC,a,b,c) \
	for(cint it=a;it;pA+=c){ it--;\
		cint jt=b;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopN(c, *pA++ OP aux**pC++);\
			pA-=c;\
		}while(jt);\
		pC-=b*c;\
	}
#define mmul_1n_plus_BC(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_bc(b,c){TypeR *pA=A, *pB=B, *pC=C;\
		mmul_1n_plus_BC_open(TypeR,pA,OP,pB,pC,a,b,c)\
	}

#define mmul_nn_equal(TypeR,A,OP,B,C,a,b,c) \
	zeroset_type(TypeR,A,a*c)\
	ZEROCHECK_ac(a,c){TypeR *pB=B;\
	TypeR *pC=C;\
	for(cint it=b;it;pC+=c){ it--;\
		TypeR *pA=A;\
		cint jt=a;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopn(c, *pA++ +##OP aux**pC++);\
			pC-=c;\
		}while(jt);\
	}}
#define mmul_nn_equal_AC(TypeR,A,OP,B,C,a,b,c) \
	zeroset_type(TypeR,A,a*c)\
	ZEROCHECK_ac(a,c){TypeR *pB=B, *pC=C, *pA=A;\
	for(cint it=b;it;pC+=c){ it--;\
		cint jt=a;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopN(c, *pA++ +##OP aux**pC++);\
			pC-=c;\
		}while(jt);\
		pA-=a*c;\
	}}

#define mmul_nn_plus(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_ac(a,c){TypeR *pB=B;\
	TypeR *pC=C;\
	for(cint it=b;it;pC+=c){ it--;\
		TypeR *pA=A;\
		cint jt=a;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopn(c, *pA++ OP aux**pC++);\
			pC-=c;\
		}while(jt);\
	}}
#define mmul_nn_plus_AC_open(TypeR,pA,OP,pB,pC,a,b,c) \
	for(cint it=b;it;pC+=c){ it--;\
		cint jt=a;\
		do{ jt--;\
			TypeR aux=*pB++;\
			inner_loopN(c, *pA++ OP aux**pC++);\
			pC-=c;\
		}while(jt);\
		pA-=a*c;\
	}
#define mmul_nn_plus_AC(TypeR,A,OP,B,C,a,b,c) \
	ZEROCHECK_ac(a,c){TypeR *pB=B, *pC=C, *pA=A;\
		mmul_nn_plus_AC_open(TypeR,pA,OP,pB,pC,a,b,c)\
	}

/* En las siguientes macros la matriz A del resultado es simétrica y las macros
solamente rellenan su mitad superior. Si se quiere tener también la parte inferior
llámese a copia_mitad(A,a) tras efectuar la multiplicación. */

//Las matrices B y C son la misma
#define automul_nn_equal_open(TypeR,A,OP,pB,a,b) \
	zeroset_type(TypeR,A,a*a)\
	for(cint it=b;it;){ it--;\
		TypeR *pA=A;\
		cint jt=a;\
		do{\
			TypeR *pC=pB;\
			TypeR aux=*pB++;\
			inner_loopn(jt--,*pA++ +##OP aux**pC++);\
			pA+=a-jt;\
		}while(jt);\
	}
#define automul_nn_equal(TypeR,A,OP,B,a,b) \
	ZEROCHECK_a(a){TypeR *pB=B;\
		automul_nn_equal_open(TypeR,A,OP,pB,a,b)\
	}

#define automul_nn_equal_open_AC(TypeR,pA,OP,pB,a,b) \
	zeroset_type(TypeR,pA,a*a)\
	for(cint it=b;it;){ it--;\
		cint jt=a;\
		do{\
			TypeR *pC=pB;\
			TypeR aux=*pB++;\
			inner_loopn(jt--,*pA++ +##OP aux**pC++);\
			pA+=a-jt;\
		}while(jt);\
		pA-=a*(a+1);\
	}
#define automul_nn_equal_AC(TypeR,A,OP,B,a,b) \
	ZEROCHECK_a(a){TypeR *pA=A, *pB=B;\
		automul_nn_equal_open_AC(TypeR,pA,OP,pB,a,b)\
	}

/* Las matrices B y C no son la misma, pero sabemos por otra razón que el
resultado es simétrico */
#define simmul_nn_plus_open(TypeR,A,OP,pB,pC,a,b) \
	for(cint it=b;it;){ it--;\
		TypeR *pA=A;\
		cint jt=a;\
		do{\
			TypeR aux=*pB++;\
			inner_loopn(jt--,*pA++ OP aux**pC++);\
			pA+=a-jt;\
			pC-=jt;\
		}while(jt);\
	}
#define simmul_nn_plus(TypeR,A,OP,B,C,a,b) \
	ZEROCHECK_a(a){TypeR *pB=B, *pC=C;\
		simmul_nn_plus_open(TypeR,A,OP,pB,pC,a,b)\
	}

/* Rellena la parte por debajo de la diagonal con los valores que hay
en la mitad superior, completando así la una matriz simétrica. */
#define matriz_copia_mitad(Tipo,N,n,nnn) \
   iflike(n){\
	Tipo *pN=N;\
	for(cint i=n-1;i;pN+=nnn){\
		Tipo *p_c, *p_b;\
		p_c=p_b=pN;\
		p_b++, p_c+=n;\
		for(cint j=i--;j--;p_b++,p_c+=n)\
			*p_c=*p_b;\
	}\
   }

/*Definiciones de mmult que escogen la variante _equal o _plus según corresponda. Véase el comentario
del principio del fichero para leer cómo me gustaría definirlas si el preprocesador de C lo permitiese.
Las definiciones de aquí añaden una rama de código inútil, pues el argumento del if() es constante.
Es de esperar que un optimizador de código elimine la rama inalcanzable.*/

#define mmul_vectorn(TypeR,A,OP,B,v,a,c) \
	if(#OP[0]=='='){mmul_vectorn_equal(TypeR,A,OP,B,v,a,c)}\
	else{mmul_vectorn_plus(TypeR,A,OP,B,v,a,c)}

#define mmult_1n(TypeR,A,OP,B,C,a,b,c) \
	if(#OP[0]=='='){mmul_1n_equal(TypeR,A,OP,B,C,a,b,c)}\
	else{mmul_1n_plus(TypeR,A,OP,B,C,a,b,c)}

#define mmult_nn(TypeR,A,OP,B,C,a,b,c) \
	if(#OP[0]=='='){mmul_nn_equal(TypeR,A,OP,B,C,a,b,c)}\
	else{mmul_nn_plus(TypeR,A,OP,B,C,a,b,c)}

#define mmult_1n_BC(TypeR,A,OP,B,C,a,b,c) \
	if(#OP[0]=='='){mmul_1n_equal_BC(TypeR,A,OP,B,C,a,b,c)}\
	else{mmul_1n_plus_BC(TypeR,A,OP,B,C,a,b,c)}

#define mmult_nn_AC(TypeR,A,OP,B,C,a,b,c) \
	if(#OP[0]=='='){mmul_nn_equal_AC(TypeR,A,OP,B,C,a,b,c)}\
	else{mmul_nn_plus_AC(TypeR,A,OP,B,C,a,b,c)}


#if COMPILER_ID!=0
//#includeif mergestring(bucles_y_matrices_,COMPILER.h)
#endif
